<?php
// Start output buffering to prevent any output before PDF generation
ob_start();

session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../vendor/autoload.php';

// Check if user has permission to export reports
if (!hasPermission('export_reports') && !$isSuperAdmin) {
    ob_end_clean(); // Clean buffer before redirect
    $_SESSION['error'] = "You don't have permission to export reports.";
    header('Location: reports.php');
    exit;
}

// Get date range from request
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'store_website', 'currency')");
$stmt->execute();
$store_settings = [];
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
}

$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';
$storeWebsite = $store_settings['store_website'] ?? '';
$currency = $store_settings['currency'] ?? 'Ksh';

// Get customer data with sales information
$stmt = $pdo->prepare("
    SELECT 
        c.id,
        c.name,
        c.phone,
        c.email,
        c.address,
        c.created_at as registration_date,
        COUNT(DISTINCT s.id) as total_orders,
        COALESCE(SUM(s.total_amount), 0) as total_spent,
        COALESCE(SUM(CASE WHEN s.payment_method = 'credit' THEN s.total_amount ELSE 0 END), 0) as credit_sales,
        COALESCE(SUM(CASE WHEN s.payment_method = 'cash' THEN s.total_amount ELSE 0 END), 0) as cash_sales,
        COALESCE(SUM(CASE WHEN s.payment_method = 'mpesa' THEN s.total_amount ELSE 0 END), 0) as mpesa_sales,
        MAX(s.created_at) as last_purchase_date,
        COALESCE(AVG(s.total_amount), 0) as average_order_value
    FROM customers c
    LEFT JOIN sales s ON c.id = s.customer_id AND DATE(s.created_at) BETWEEN ? AND ?
    GROUP BY c.id, c.name, c.phone, c.email, c.address, c.created_at
    ORDER BY total_spent DESC, c.name ASC
");
$stmt->execute([$start_date, $end_date]);
$customers = $stmt->fetchAll();

// Get overall customer statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT c.id) as total_customers,
        COUNT(DISTINCT CASE WHEN s.id IS NOT NULL THEN c.id END) as active_customers,
        COUNT(DISTINCT CASE WHEN s.payment_method = 'credit' THEN c.id END) as credit_customers,
        COALESCE(SUM(s.total_amount), 0) as total_revenue,
        COALESCE(AVG(s.total_amount), 0) as average_order_value,
        COUNT(s.id) as total_transactions
    FROM customers c
    LEFT JOIN sales s ON c.id = s.customer_id AND DATE(s.created_at) BETWEEN ? AND ?
");
$stmt->execute([$start_date, $end_date]);
$stats = $stmt->fetch();

// Create PDF using TCPDF
class MYPDF extends TCPDF {
    public function Header() {
        global $storeName, $storeAddress, $storePhone, $storeEmail, $storeWebsite, $start_date, $end_date;
        // Report title first
        $this->SetFont('helvetica', 'B', 16);
        $this->Cell(0, 10, 'CUSTOMERS ACTIVITY REPORT', 0, 1, 'C');
        // Store information
        $this->SetFont('helvetica', 'B', 14);
        $this->Cell(0, 10, $storeName, 0, 1, 'C');
        $this->SetFont('helvetica', '', 10);
        $this->Cell(0, 5, $storeAddress, 0, 1, 'C');
        $this->Cell(0, 5, 'Phone: ' . $storePhone . ' | Email: ' . $storeEmail, 0, 1, 'C');
        if (!empty($storeWebsite)) {
            $this->Cell(0, 5, 'Website: ' . $storeWebsite, 0, 1, 'C');
        }
        $this->Ln(5);
    }
    
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('helvetica', 'I', 8);
        $this->Cell(0, 10, 'Generated on ' . date('Y-m-d H:i:s') . ' | Page ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, 0, 'C');
    }
}

// Create new PDF document
$pdf = new MYPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

// Set document information
$pdf->SetCreator('POS System');
$pdf->SetAuthor($storeName);
$pdf->SetTitle('Customers Report - ' . $start_date . ' to ' . $end_date);
$pdf->SetSubject('Customer Activity Report');

// Set margins (1.5cm left and right)
$pdf->SetMargins(15, 30, 15); // 15mm = 1.5cm
$pdf->SetHeaderMargin(5);
$pdf->SetFooterMargin(10);

// Set auto page breaks
$pdf->SetAutoPageBreak(TRUE, 25);

// Add a page
$pdf->AddPage();

// Add 1.5cm (15mm) vertical space after header
$pdf->Ln(15);

// Line before Report Period
$pdf->SetDrawColor(0,0,0);
$pdf->Line($pdf->GetX(), $pdf->GetY(), $pdf->getPageWidth() - $pdf->GetX(), $pdf->GetY());
$pdf->Ln(2);

// Report Period (left-aligned, before Customer Overview)
$pdf->SetFont('helvetica', 'B', 12);
$pdf->Cell(0, 8, 'Report Period: ' . date('M d, Y', strtotime($start_date)) . ' to ' . date('M d, Y', strtotime($end_date)), 0, 1, 'L');

// Line after Report Period
$pdf->Line($pdf->GetX(), $pdf->GetY(), $pdf->getPageWidth() - $pdf->GetX(), $pdf->GetY());
$pdf->Ln(5);

// Summary statistics
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'CUSTOMER OVERVIEW', 0, 1, 'L');
$pdf->Ln(2);

$pdf->SetFont('helvetica', '', 10);
$pdf->Cell(50, 6, 'Total Customers:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['total_customers']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Active Customers:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['active_customers']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Credit Customers:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['credit_customers']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Total Revenue:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['total_revenue'], 2), 0, 1, 'L');

$pdf->Cell(50, 6, 'Total Transactions:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['total_transactions']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Avg Order Value:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['average_order_value'], 2), 0, 1, 'L');

$pdf->Ln(10);

// Detailed customer information
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'DETAILED CUSTOMER INFORMATION', 0, 1, 'L');
$pdf->Ln(2);

// Table header
$pdf->SetFont('helvetica', 'B', 8);
$pdf->SetFillColor(230, 230, 230);
$pdf->Cell(35, 8, 'Customer Name', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Phone', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Orders', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Total Spent', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Cash', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'M-Pesa', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Credit', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Last Purchase', 1, 1, 'C', true);

// Table data
$pdf->SetFont('helvetica', '', 7);

foreach ($customers as $customer) {
    // Check if we need a new page
    if ($pdf->GetY() > 250) {
        $pdf->AddPage();
        // Re-add table header
        $pdf->SetFont('helvetica', 'B', 8);
        $pdf->SetFillColor(230, 230, 230);
        $pdf->Cell(35, 8, 'Customer Name', 1, 0, 'C', true);
        $pdf->Cell(25, 8, 'Phone', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Orders', 1, 0, 'C', true);
        $pdf->Cell(25, 8, 'Total Spent', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Cash', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'M-Pesa', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Credit', 1, 0, 'C', true);
        $pdf->Cell(25, 8, 'Last Purchase', 1, 1, 'C', true);
        $pdf->SetFont('helvetica', '', 7);
    }
    
    // Color code based on customer activity
    if ($customer['total_orders'] == 0) {
        $pdf->SetFillColor(255, 230, 230); // Light red for inactive customers
    } elseif ($customer['credit_sales'] > 0) {
        $pdf->SetFillColor(255, 255, 200); // Yellow for credit customers
    } else {
        $pdf->SetFillColor(255, 255, 255); // White for regular customers
    }
    
    $pdf->Cell(35, 6, substr($customer['name'], 0, 20), 1, 0, 'L', true);
    $pdf->Cell(25, 6, substr($customer['phone'] ?? 'N/A', 0, 15), 1, 0, 'L', true);
    $pdf->Cell(20, 6, number_format($customer['total_orders']), 1, 0, 'C', true);
    $pdf->Cell(25, 6, $currency . ' ' . number_format($customer['total_spent'], 2), 1, 0, 'R', true);
    $pdf->Cell(20, 6, number_format($customer['cash_sales'], 0), 1, 0, 'R', true);
    $pdf->Cell(20, 6, number_format($customer['mpesa_sales'], 0), 1, 0, 'R', true);
    $pdf->Cell(20, 6, number_format($customer['credit_sales'], 0), 1, 0, 'R', true);
    $pdf->Cell(25, 6, $customer['last_purchase_date'] ? date('Y-m-d', strtotime($customer['last_purchase_date'])) : 'Never', 1, 1, 'C', true);
}

// Add legend
$pdf->Ln(5);
$pdf->SetFont('helvetica', 'B', 9);
$pdf->Cell(0, 6, 'LEGEND:', 0, 1, 'L');
$pdf->SetFont('helvetica', '', 8);

$pdf->SetFillColor(255, 230, 230);
$pdf->Cell(15, 5, '', 1, 0, 'C', true);
$pdf->Cell(40, 5, 'Inactive Customer', 0, 0, 'L');

$pdf->SetFillColor(255, 255, 200);
$pdf->Cell(15, 5, '', 1, 0, 'C', true);
$pdf->Cell(40, 5, 'Credit Customer', 0, 0, 'L');

$pdf->SetFillColor(255, 255, 255);
$pdf->Cell(15, 5, '', 1, 0, 'C', true);
$pdf->Cell(40, 5, 'Regular Customer', 0, 1, 'L');

// Generate filename
$filename = 'Customers_Report_' . $start_date . '_to_' . $end_date . '.pdf';

// Create absolute path for reports directory
$reports_dir = dirname(__DIR__) . '/Reports/';
if (!file_exists($reports_dir)) {
    mkdir($reports_dir, 0755, true);
}

// Save PDF to reports folder (optional - for record keeping)
$filepath = $reports_dir . $filename;
try {
    $pdf->Output($filepath, 'F');
} catch (Exception $e) {
    // If saving fails, continue with download only
    error_log("Failed to save PDF to reports folder: " . $e->getMessage());
}

// Clean any output buffer before sending PDF
ob_end_clean();

// Output PDF to browser for download
$pdf->Output($filename, 'D');
exit;
?> 